#!/bin/bash
# dotfiles_backup.sh
#
# What this does (high level):
# - Backs up your ~/.config directory to a NAS location using rsync
# - Also creates a timestamped tar.gz archive of ~/.config on the NAS
# - Tries a "local mount" first; if not mounted, it can bring up PiVPN and retry
# - Coordinates with PIA VPN so routing doesn’t block NAS access:
#     - If PIA was connected at the start, it disconnects it before mounting/backup
#     - After backup + unmount, it restores PIA if it was originally connected
#
# What it depends on:
# - Your NAS mount script (MOUNT_SCRIPT) that mounts all required NFS shares
# - PiVPN connect script (pivpn-connect.sh) + systemd OpenVPN unit
# - Optional: PIA CLI (piactl) for VPN coordination
# - rsync, tar, mountpoint, sudo, systemctl
#
# Publish note / privacy:
# - PRIVATE: NAS paths, mountpoints, and backup directory names are obfuscated.
# - The exclude list (BraveSoftware/, Element/, mise/) is not sensitive.
# - No passwords or tokens are stored here.

set -euo pipefail

echo "=== dotfiles-backup: starting ==="

# --- Paths / services ---

# PRIVATE: NAS mountpoint (obfuscated for publishing)
NAS_MOUNT="/mnt/NAS_MOUNT_HERE"

# PRIVATE: backup directory on the NAS (obfuscated for publishing)
NAS_DOTFILES_DIR="$NAS_MOUNT/backups/NAS_DOTFILES_BACKUP_DIR"

# Source directory to back up
SRC_CONFIG="$HOME/.config"

# All NAS mounts your mount_nfs.sh creates
# PRIVATE: mountpoints obfuscated for publishing
NAS_MOUNTS=(
  "/mnt/MOUNT_POINT_1"
  "/mnt/MOUNT_POINT_2"
  "/mnt/MOUNT_POINT_3"
  "/mnt/MOUNT_POINT_4"
)

# PRIVATE: mount script path/name (obfuscated for publishing)
MOUNT_SCRIPT="$HOME/MOUNT_SCRIPT_HERE.sh"

PIVPN_SERVICE="openvpn-client@pivpn"
PIVPN_CONNECT="$HOME/.local/bin/pivpn-connect.sh"

PIACTL="/opt/piavpn/bin/piactl"

# Remember original PIA state so we can restore it at the end
INITIAL_PIA_STATE="Unknown"
if [[ -x "$PIACTL" ]]; then
  INITIAL_PIA_STATE="$("$PIACTL" get connectionstate 2>/dev/null || echo "Unknown")"
  echo "[dotfiles-backup] Initial PIA state: $INITIAL_PIA_STATE"
else
  echo "[dotfiles-backup] PIA client not found at $PIACTL – skipping PIA handling."
fi

# --- Helper functions ---

try_mount() {
  echo "[dotfiles-backup] Attempting to mount NAS via $MOUNT_SCRIPT..."

  if [[ -x "$MOUNT_SCRIPT" ]]; then
    "$MOUNT_SCRIPT"
  else
    echo "[dotfiles-backup] ERROR: $MOUNT_SCRIPT not found or not executable." >&2
    return 1
  fi
}

ensure_pivpn() {
  if systemctl is-active --quiet "$PIVPN_SERVICE"; then
    echo "[dotfiles-backup] PiVPN is already active (service $PIVPN_SERVICE)."
    return 0
  fi

  echo "[dotfiles-backup] PiVPN is not active. Trying to bring it up with $PIVPN_CONNECT..."

  if [[ -x "$PIVPN_CONNECT" ]]; then
    # This may pop your polkit auth window (fingerprint/password), which is expected.
    "$PIVPN_CONNECT" >/dev/null 2>&1 || {
      echo "[dotfiles-backup] WARNING: pivpn-connect.sh returned a non-zero status." >&2
      return 1
    }
    echo "[dotfiles-backup] PiVPN connect script was invoked."
  else
    echo "[dotfiles-backup] ERROR: $PIVPN_CONNECT not found or not executable; cannot use PiVPN." >&2
    return 1
  fi
}

ensure_pia_disconnected() {
  # If PIA tools aren't present, nothing to do
  if [[ ! -x "$PIACTL" ]]; then
    return 0
  fi

  # Only disconnect if it *was* connected initially
  if [[ "$INITIAL_PIA_STATE" == "Connected" || "$INITIAL_PIA_STATE" == "Connecting" ]]; then
    echo "[dotfiles-backup] PIA was $INITIAL_PIA_STATE at start – disconnecting it so NAS is reachable..."
    "$PIACTL" disconnect || true
    sleep 3
  else
    echo "[dotfiles-backup] PIA was not connected at start – leaving it alone."
  fi
}

cleanup_nas_mounts() {
  echo "[dotfiles-backup] Cleaning up NAS mounts before finishing..."

  for m in "${NAS_MOUNTS[@]}"; do
    if mountpoint -q "$m"; then
      echo "  - Unmounting $m ..."
      # This may trigger a polkit auth prompt if sudo is needed
      sudo umount "$m" || echo "    ! Failed to unmount $m (you may need to check it manually)."
    fi
  done

  echo "[dotfiles-backup] NAS mount cleanup done."
}

restore_pia_if_needed() {
  if [[ ! -x "$PIACTL" ]]; then
    echo "[dotfiles-backup] PIA client not installed – nothing to restore."
    return
  fi

  if [[ "$INITIAL_PIA_STATE" == "Connected" || "$INITIAL_PIA_STATE" == "Connecting" ]]; then
    echo "[dotfiles-backup] Restoring PIA to its original state (connecting)..."
    "$PIACTL" connect || true
  else
    echo "[dotfiles-backup] PIA was not connected originally – not reconnecting."
  fi
}

fail_and_exit() {
  local msg="$1"
  echo "[dotfiles-backup] ERROR: $msg" >&2
  echo "=== dotfiles-backup: FAILED ==="
  # Try to clean up mounts if they exist
  cleanup_nas_mounts
  # Restore PIA if it was originally on
  restore_pia_if_needed
  exit 1
}

# --- 0) Handle PIA first so NAS is reachable ---
echo "[dotfiles-backup] Checking PIA VPN state..."
ensure_pia_disconnected

# --- 1) Check if NAS mount is available / mount it ---

echo "[dotfiles-backup] Checking if NAS mount ($NAS_MOUNT) is available..."

# Try local mount first (no VPN)
if ! mountpoint -q "$NAS_MOUNT"; then
  echo "[dotfiles-backup] $NAS_MOUNT is not mounted."
  echo "[dotfiles-backup] First, trying a normal local mount (no VPN)..."
  try_mount || echo "[dotfiles-backup] Local mount attempt failed or script returned non-zero."
fi

# If still not mounted, try via PiVPN
if ! mountpoint -q "$NAS_MOUNT"; then
  echo "[dotfiles-backup] NAS still not mounted. Assuming we might be remote."
  echo "[dotfiles-backup] Trying to bring up PiVPN tunnel and mount over it..."

  if ensure_pivpn; then
    echo "[dotfiles-backup] PiVPN should now be active. Retrying NAS mount..."
    try_mount || echo "[dotfiles-backup] Mount attempt over PiVPN returned non-zero."
  else
    fail_and_exit "Could not start PiVPN; cannot reach NAS remotely."
  fi
fi

# Final check
if ! mountpoint -q "$NAS_MOUNT"; then
  fail_and_exit "$NAS_MOUNT is still not mounted after local + PiVPN attempts. Is the NAS offline or PiVPN unreachable?"
fi

echo "[dotfiles-backup] NAS is mounted at $NAS_MOUNT ✅"

# --- 1.5) Verify NAS backup directory exists ---

echo "[dotfiles-backup] Verifying that $NAS_DOTFILES_DIR exists on the NAS..."

if [[ ! -d "$NAS_DOTFILES_DIR" ]]; then
  fail_and_exit "Expected NAS backup directory $NAS_DOTFILES_DIR does not exist. Create it and try again."
fi

echo "[dotfiles-backup] Found existing NAS backup directory: $NAS_DOTFILES_DIR ✅"

# --- 2) Rsync dotfiles to NAS ---

DEST_CONFIG="$NAS_DOTFILES_DIR/.config"
mkdir -p "$DEST_CONFIG"
echo "[dotfiles-backup] Preparing destination: $DEST_CONFIG"

echo "[dotfiles-backup] Rsyncing ~/.config -> $DEST_CONFIG"
echo "[dotfiles-backup] Excluding: BraveSoftware/, Element/, mise/"

rsync -aAHv \
  --exclude='BraveSoftware/' \
  --exclude='Element/' \
  --exclude='mise/' \
  "$SRC_CONFIG/" \
  "$DEST_CONFIG/"

echo "[dotfiles-backup] Rsync complete ✅"

# --- 3) Create tarball on NAS ---

TAR_PATH="$NAS_DOTFILES_DIR/dotfiles-config-$(date +%Y%m%d-%H%M%S).tar.gz"

echo "[dotfiles-backup] Creating tarball at:"
echo "  $TAR_PATH"
echo "[dotfiles-backup] (Excluding BraveSoftware, Element, mise)"

tar czf "$TAR_PATH" \
  --exclude='.config/BraveSoftware' \
  --exclude='.config/Element' \
  --exclude='.config/mise' \
  -C "$HOME" .config

echo "[dotfiles-backup] Tarball created ✅"

# --- 4) Unmount NAS mounts before PIA comes back ---

cleanup_nas_mounts

# --- 5) Restore PIA if needed ---

restore_pia_if_needed

echo "[dotfiles-backup] All done."
echo "[dotfiles-backup] Directory backup: $DEST_CONFIG"
echo "[dotfiles-backup] Archive backup:   $TAR_PATH"
echo "=== dotfiles-backup: finished successfully ==="

