#!/usr/bin/env bash
# pivpn-disconnect.sh
#
# What this does:
# - Disconnects your PiVPN OpenVPN client via systemd:
#     openvpn-client@pivpn
# - Optionally restores PIA VPN if it was connected before you enabled PiVPN:
#     - pivpn-connect.sh writes /tmp/pivpn-pia-was-connected when it had to
#       disconnect PIA to bring PiVPN up
#     - If that state file exists here, this script reconnects PIA after
#       stopping PiVPN, then removes the state file

set -euo pipefail

# OpenVPN systemd unit for PiVPN
SERVICE="openvpn-client@pivpn"

# PIA CLI path (used to restore PIA after PiVPN is stopped)
PIACTL="/opt/piavpn/bin/piactl"

# State flag file created by pivpn-connect.sh if it disconnected PIA
STATE_FILE="/tmp/pivpn-pia-was-connected"

log() {
  # Only print if stdout is a terminal
  if [[ -t 1 ]]; then
    echo "$@"
  fi
}

log "[PiVPN] Disconnecting…"

# 1) Stop PiVPN via systemd (polkit will handle auth if needed)
if systemctl is-active --quiet "$SERVICE"; then
  if ! systemctl stop "$SERVICE"; then
    log "[PiVPN] ERROR: failed to stop $SERVICE."
    exit 1
  fi
else
  log "[PiVPN] $SERVICE is not active."
fi

log "[PiVPN] Disconnected."

# 2) If PIA was previously connected, reconnect it now
if [[ -x "$PIACTL" && -f "$STATE_FILE" ]]; then
  log "[PiVPN] PIA was connected before PiVPN. Reconnecting PIA…"
  "$PIACTL" connect || true
  rm -f "$STATE_FILE"
fi

