#!/usr/bin/env bash
# pivpn-connect.sh
#
# What this does:
# - Connects your PiVPN OpenVPN client via systemd:
#     openvpn-client@pivpn
# - Coordinates with PIA VPN (if installed) to avoid conflicts:
#     - If PIA is Connected/Connecting, it disconnects PIA first
#     - It writes a small state flag so the disconnect script can later restore PIA
#
# Key behavior:
# - Writes /tmp/pivpn-pia-was-connected when it *actually* had to disconnect PIA
# - If PiVPN is already active, it exits cleanly

set -euo pipefail

# OpenVPN systemd unit for PiVPN
SERVICE="openvpn-client@pivpn"

# PIA CLI path (used to disconnect PIA before bringing PiVPN up)
PIACTL="/opt/piavpn/bin/piactl"

# State flag file used to remember whether PIA was connected before PiVPN connect
STATE_FILE="/tmp/pivpn-pia-was-connected"

log() {
  # Only print if stdout is a terminal
  if [[ -t 1 ]]; then
    echo "$@"
  fi
}

log "[PiVPN] Connecting…"

# 1) Handle PIA state
# If PIA is connected (or connecting), disconnect it first and remember that fact.
if [[ -x "$PIACTL" ]]; then
  state="$("$PIACTL" get connectionstate 2>/dev/null || true)"
  if [[ "$state" == "Connected" || "$state" == "Connecting" ]]; then
    log "[PiVPN] PIA is $state – disconnecting it first…"
    echo "1" > "$STATE_FILE"          # <-- real file write, no log()
    "$PIACTL" disconnect || true
    sleep 2
  else
    rm -f "$STATE_FILE"
  fi
else
  # If piactl isn't installed/executable, ensure we don't leave stale state behind
  rm -f "$STATE_FILE"
fi

# 2) If PiVPN already active, bail
if systemctl is-active --quiet "$SERVICE"; then
  log "[PiVPN] $SERVICE is already active."
  exit 0
fi

# 3) Start via systemd (polkit will handle auth if needed)
log "[PiVPN] Starting $SERVICE via systemd…"
if ! systemctl start "$SERVICE"; then
  log "[PiVPN] ERROR: failed to start $SERVICE."
  exit 1
fi

log "[PiVPN] Started."

