#!/usr/bin/env bash
# pia-pick-region.sh
#
# What this does:
# - Fetches available Private Internet Access (PIA) regions using piactl
# - Lets you pick a region using a simple UI picker:
#     - rofi (preferred)
#     - wofi
#     - zenity (GUI fallback)
# - Sets the selected region via:
#     piactl set region "<choice>"
# - Sends a desktop notification confirming the selection
#
# Dependencies / assumptions:
# - PIA is installed and `piactl` exists at /opt/piavpn/bin/piactl
# - You are logged in to PIA (otherwise regions may be empty)
# - One of rofi/wofi/zenity is installed for selection UI

PIACTL="/opt/piavpn/bin/piactl"

# Get regions from piactl, then normalize to one region per line
regions="$($PIACTL get regions 2>/dev/null | tr ' ' '\n' | sed '/^$/d')"
[ -z "$regions" ] && notify-send "PIA" "No regions yet. Try: piactl login" && exit 1

# Pick with rofi/wofi, fallback to zenity
if command -v rofi >/dev/null; then
  choice="$(echo "$regions" | rofi -dmenu -p 'PIA region')"
elif command -v wofi >/dev/null; then
  choice="$(echo "$regions" | wofi --dmenu -p 'PIA region')"
elif command -v zenity >/dev/null; then
  choice="$(echo "$regions" | zenity --list --column=Region)"
else
  notify-send "PIA" "Install rofi/wofi/zenity to pick regions."
  exit 1
fi

# If user cancelled, exit cleanly
[ -z "$choice" ] && exit 0

# Apply selected region and notify
$PIACTL set region "$choice" && notify-send "PIA" "Region set: $choice"

